import 'package:flutter/material.dart';
import 'package:flutter_bloc/flutter_bloc.dart';
import 'package:hamlemuhcrm/features/auth/login/cubit/login_cubit.dart';
import 'package:hamlemuhcrm/features/projects/view/missions_partial/utility/custom_auto_complete.dart';
import '../../../../core/constants/app_constants.dart';
import '../../../../core/constants/color_constants.dart';
import '../../../../core/constants/font_styles.dart';
import '../../../../core/extensions/responsive_extension.dart';
import '../../cubit/mission_employee_notes_cubit.dart';
import '../../model/mission_employee_note.dart';
import 'widgets/employee_note_card.dart';

class MainMissionEmployeeNotes extends StatefulWidget {
  final String mainMissionName;
  final String mainMissionId;
  final bool isEditNote;

  const MainMissionEmployeeNotes({
    super.key,
    required this.mainMissionId,
    required this.mainMissionName,
    required this.isEditNote,
  });

  @override
  State<MainMissionEmployeeNotes> createState() =>
      MainMissionEmployeeNotesState();
}

class MainMissionEmployeeNotesState extends State<MainMissionEmployeeNotes> {
  @override
  void initState() {
    context.read<MissionEmployeeCubit>().fetchEmployeeNotes(
        context.read<LoginCubit>().token,
        context.read<LoginCubit>().id,
        widget.mainMissionId);
    super.initState();
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: _appBar(),
      body: Padding(
        padding: _pagePaddingHorizontal,
        child: Column(
          children: [
            _tfContainer('Görev Notları'),
            _emptyHeightSpaceNormal,
            Visibility(
                visible: !(MediaQuery.viewInsetsOf(context).bottom > 0),
                child: _mainContainer(_suggestionCubit(context))),
            _emptyHeightSpaceNormal,
            Visibility(
                visible: widget.isEditNote, child: _mainContainer(_tfNote)),
            _emptyHeightSpaceNormal,
          ],
        ),
      ),
    );
  }

  AppBar _appBar() {
    return AppBar(
      backgroundColor: Colors.transparent,
      elevation: 0,
      title: Text(gorevAdKucult(widget.mainMissionName),
          style: FontStyles.instance.appbarTitleStyle),
      centerTitle: true,
      iconTheme: const IconThemeData(color: Colors.black87),
    );
  }

  String gorevAdKucult(String toplantiBaslik) => toplantiBaslik.length > 32
      ? toplantiBaslik.substring(0, 31)
      : toplantiBaslik;

  EdgeInsets get _pagePaddingHorizontal =>
      EdgeInsets.symmetric(horizontal: context.dynamicWidth(0.06));

  Widget get _emptyHeightSpaceNormal => SizedBox(
        height: context.dynamicHeight(0.03),
      );

  Widget _tfContainer(String headline) => Expanded(
        flex: 1,
        child: Container(
          height: context.dynamicHeight(0.05),
          decoration: BoxDecoration(
              color: ColorsConstants.instance.baseTheme,
              borderRadius: BorderRadius.circular(6)),
          child: Center(
            child: Text(headline,
                style: FontStyles.instance.girisBtnYazi.copyWith(fontSize: 16)),
          ),
        ),
      );

  Widget get _tfNote => Padding(
      padding: const EdgeInsets.all(5),
      child: CustomAutoComplete(isGorevId: widget.mainMissionId));

  Widget _mainContainer(Widget child) => Expanded(
        flex: 10,
        child: Container(
          decoration: BoxDecoration(
            color: ColorsConstants.instance.backgroundColor,
            borderRadius: BorderRadius.circular(
                ApplicationConstants.instance.kBorderRadius),
            border: Border.all(
                color: ColorsConstants.instance.iconColor, width: 1.2),
          ),
          child: child,
        ),
      );

  Widget _suggestionCubit(BuildContext context) {
    return BlocBuilder<MissionEmployeeCubit, MissionEmployeeState>(
      builder: (context, state) {
        if (state.calisanNotlari.isEmpty &&
            !state.isCompleted &&
            !state.isLoading &&
            !state.isError &&
            !state.isLoaded) {
          return Center(
            child: Text(
              'Boş Liste',
              style: Theme.of(context).textTheme.headlineSmall,
            ),
          );
        } else if (state.isLoading & !state.isCompleted &&
            !state.isLoaded &&
            !state.isError) {
          return const Center(
            child: CircularProgressIndicator(),
          );
        } else if (state.isCompleted & !state.isLoading && !state.isError) {
          final calisanNotListe = state.calisanNotlari;
          return _listViewSuggestions(calisanNotListe);
        } else {
          return const Center(
            child: Text('Hata Oluştu'),
          );
        }
      },
    );
  }

  Widget _listViewSuggestions(List<MissionEmployeeNote> calisanList) =>
      BlocBuilder<MissionEmployeeCubit, MissionEmployeeState>(
          builder: (context, state) {
        return ListView.builder(
            itemCount: calisanList.length,
            itemBuilder: (context, index) {
              return EmployeeNoteCard(
                model: calisanList[index],
              );
            });
      });
}
