import 'package:flutter/material.dart';
import 'package:flutter_bloc/flutter_bloc.dart';
import 'package:hamlemuhcrm/features/projects/view/projects_partial/project_content_page.dart';

import '../../../core/constants/color_constants.dart';
import '../../../core/extensions/responsive_extension.dart';
import '../../../core/route/page_route.dart';
import '../cubit/project_cubit.dart';
import '../cubit/project_state.dart';
import '../model/projeler.dart';
import '../view/projects_partial/project_create_page.dart';
import '../widgets/filter_project_chip_list.dart';
import '../widgets/project_card.dart';

class ProjectTabBar extends StatelessWidget {
  const ProjectTabBar({super.key});

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: PreferredSize(
          preferredSize: Size.fromHeight(context.dynamicHeight(0.065)),
          child: _appbar(context)),
      body: _projectStateCubit(context),
    );
  }

  Widget _appbar(BuildContext context) {
    return AppBar(
      backgroundColor: Colors.transparent,
      elevation: 0,
      actions: [
        _addProjectButton(context),
      ],
      title: const FilterProjectChip(),
    );
  }

  Widget _scrollBar(Widget child) {
    return Scrollbar(
      thickness: 6,
      radius: const Radius.circular(10),
      child: child,
    );
  }

  Widget _addProjectButton(BuildContext context) => Padding(
        padding: EdgeInsets.symmetric(horizontal: context.dynamicWidth(0.03)),
        child: Row(
          mainAxisAlignment: MainAxisAlignment.spaceBetween,
          children: [
            GestureDetector(
              onTap: () {
                Navigator.push(
                    context, CustomPageRoute(child: const ProjectCreatePage()));
              },
              child: Container(
                height: context.dynamicHeight(0.04),
                width: context.dynamicWidth(0.3),
                decoration: BoxDecoration(
                  color: Colors.white,
                  boxShadow: [
                    BoxShadow(
                        color: Colors.grey.withOpacity(0.3),
                        blurRadius: 4,
                        offset: const Offset(1.5, 3))
                  ],
                  borderRadius: BorderRadius.circular(10),
                ),
                child: Row(
                  mainAxisAlignment: MainAxisAlignment.spaceEvenly,
                  children: [
                    Icon(
                      Icons.add_circle,
                      size: context.dynamicHeight(0.03),
                      color: ColorsConstants.instance.baseTheme,
                    ),
                    const Text('Proje Oluştur',
                        style: TextStyle(color: Colors.black, fontSize: 13)),
                  ],
                ),
              ),
            ),
          ],
        ),
      );

  Widget _projectListView(BuildContext context, List<Projeler> projeListesi) {
    return SizedBox(
      height: context.dynamicHeight(1),
      child: _scrollBar(
        ListView.builder(
          itemCount: projeListesi.length,
          shrinkWrap: true,
          itemBuilder: (context, index) {
            final proje = projeListesi[index];
            return ProjectCard(
              model: proje,
              onTap: () {
                Navigator.push(
                    context,
                    CustomPageRoute(
                        child: ProjectContentPage(projeler: proje)));
              },
            );
          },
        ),
      ),
    );
  }

  Widget _projectStateCubit(BuildContext context) {
    return BlocBuilder<ProjectCubit, ProjectState>(
      builder: (context, state) {
        if (state.status == ProjectStatus.initial) {
          return const Center(
            child: Text('Yükleniyor..'),
          );
        } else if (state.status == ProjectStatus.loading) {
          return const Center(
            child: CircularProgressIndicator(),
          );
        } else if (state.status == ProjectStatus.error) {
          return Center(
            child: Text(state.errorMessage),
          );
        } else if (state.projeListesi.isEmpty) {
          return Center(
            child: Text(
              'Proje Bulunamadı',
              style: Theme.of(context).textTheme.titleMedium,
            ),
          );
        } else {
          final projeListesi = state.projeListesi;
          return _projectListView(context, projeListesi);
        }
      },
    );
  }
}
