import 'dart:math';

import 'package:flutter/material.dart';
import 'package:hamlemuhcrm/core/constants/app_constants.dart';
import 'package:hamlemuhcrm/core/constants/color_constants.dart';
import 'package:hamlemuhcrm/core/extensions/responsive_extension.dart';

import '../../../core/constants/url_services_constant.dart';
import '../constants/suggestion_constant.dart';

class SuggestionCardItem extends StatelessWidget {
  final String meetTitle;
  final String date;
  final String profileImage;
  final String suggestedName;
  final void Function()? onPressedConfirm;
  final void Function()? onPressedDeleted;
  final bool isVerified;
  final String durumVeOnay;

  const SuggestionCardItem({
    super.key,
    required this.meetTitle,
    required this.date,
    required this.profileImage,
    required this.isVerified,
    this.onPressedConfirm,
    this.onPressedDeleted,
    // optional parameter verirse değerini alır vermez ise default değer gözükür
    this.suggestedName = 'SOFTRUE',
    required this.durumVeOnay,
  });

  @override
  Widget build(BuildContext context) {
    return _mainContainer(
      context,
      Row(
        children: [
          _startEdgeContainer(context),
          _emptySpaceWidth(context),
          Expanded(
            child: Column(
              mainAxisAlignment: MainAxisAlignment.center, //kartı ortaladık
              children: [
                _upHeaderZone(context),
                _seperatorDivider,
                _downHeaderZone(context),
              ],
            ),
          ),
          _emptySpaceWidth(context),
        ],
      ),
    );
  }

  Widget _mainContainer(BuildContext context, Widget child) {
    return Container(
        height: context.dynamicHeight(0.18),
        margin: _marginContainerAll,
        decoration: BoxDecoration(
          borderRadius: BorderRadius.circular(_kborderRadiusValue),
          border: Border.all(
              color: ColorsConstants.instance.containerBorderGreyColor
                  .withOpacity(0.5),
              width: 1.0),
          boxShadow: [
            BoxShadow(
                color: Colors.grey.shade200,
                blurRadius: 6,
                offset: const Offset(2, 3))
          ],
          color: Colors.white,
        ),
        child: child);
  }

  Widget get _seperatorDivider => Divider(
        height: _kDividerHeigth,
        color:
            ColorsConstants.instance.containerBorderGreyColor.withOpacity(0.8),
      );

  Widget _upHeaderZone(BuildContext context) => Row(
        mainAxisAlignment: MainAxisAlignment.spaceBetween,
        children: [
          Column(
            crossAxisAlignment: CrossAxisAlignment.start,
            mainAxisAlignment: MainAxisAlignment.center,
            children: [
              Text(
                titleKucult(meetTitle),
                style: Theme.of(context)
                    .textTheme
                    .labelMedium
                    ?.copyWith(fontWeight: FontWeight.w600),
              ),
              _emptySpaceHeight(context),
              Row(mainAxisAlignment: MainAxisAlignment.spaceEvenly, children: [
                Icon(Icons.free_cancellation_rounded,
                    color: ColorsConstants.instance.baseTheme,
                    size: _kCalendarIconSize),
                _emptySpaceWidth(context),
                Text(
                  date,
                  style: Theme.of(context).textTheme.labelMedium,
                ),
                const SizedBox(
                  width: 15,
                ),
                Text(
                  durumVeOnay,
                  style: Theme.of(context).textTheme.bodyLarge,
                ),
              ]),
            ],
          ),
          isVerified ? _messageIcon : _buttonList,
        ],
      );

  Widget _downHeaderZone(BuildContext context) => Row(
        mainAxisAlignment: MainAxisAlignment.spaceBetween,
        children: [
          Column(
            crossAxisAlignment: CrossAxisAlignment.start,
            children: [
              Text(
                SuggestionConstants.instance.bySuggestionDescriptionTxt,
                style: Theme.of(context)
                    .textTheme
                    .labelMedium
                    ?.copyWith(fontWeight: FontWeight.w600),
              ),
              _emptySpaceHeight(context),
              Text(
                suggestedName,
                style: Theme.of(context).textTheme.labelMedium,
              ),
            ],
          ),
          _profileImage(context),
        ],
      );

  String titleKucult(String gelenTitle) =>
      gelenTitle.length > 28 ? '${gelenTitle.substring(0, 27)}..' : gelenTitle;

  Widget get _messageIcon => CircleAvatar(
        radius: 20,
        backgroundColor: Colors.transparent,
        child: Icon(Icons.message_rounded,
            color: ColorsConstants.instance.baseTheme, size: 29),
      );

  Widget get _buttonList => Row(
        children: [_agreeButton, _deleteButton],
      );

  Widget get _agreeButton => IconButton(
      padding: EdgeInsets.zero,
      iconSize: _kMiniButtonHeight,
      onPressed: onPressedConfirm,
      icon: Icon(
        Icons.check_circle_rounded,
        color: ColorsConstants.instance.appGreen,
      ));

  Widget get _deleteButton => IconButton(
      padding: EdgeInsets.zero,
      iconSize: _kMiniButtonHeight,
      onPressed: onPressedDeleted,
      icon: Icon(
        Icons.delete_forever_rounded,
        color: ColorsConstants.instance.appRed,
      ));

  double get _kMiniButtonHeight => 25.0;

  Widget _emptySpaceWidth(BuildContext context) => SizedBox(
        width: context.dynamicWidth(0.03),
      );

  Widget _emptySpaceHeight(BuildContext context) => SizedBox(
        height: context.dynamicHeight(0.005),
      );

  Widget _startEdgeContainer(BuildContext context) => Container(
        height: context.dynamicHeight(0.18),
        width: ApplicationConstants.instance.kEdgeStartContainerWidthLow,
        decoration: BoxDecoration(
          borderRadius: BorderRadius.only(
              topLeft: Radius.circular(_kborderRadiusValue - 1),
              bottomLeft: Radius.circular(_kborderRadiusValue - 1)),
          color: Color((Random().nextDouble() * 0xFFFFFF).toInt())
              .withOpacity(1.0),
        ),
      );

  Widget _profileImage(BuildContext context) => CircleAvatar(
        radius: 20,
        backgroundColor: Colors.grey.shade200,
        child: CircleAvatar(
          radius: 18,
          backgroundImage: NetworkImage(
              '${URLAPIService.instance.fileBaseURL}/upload-user/$profileImage'),
          onBackgroundImageError: (exception, stackTrace) =>
              const Icon(Icons.image_not_supported_outlined),
          backgroundColor: Colors.grey.shade100,
        ),
      );

  double get _kborderRadiusValue => 5.0;
  double get _kDividerHeigth => 15.0;
  double get _kCalendarIconSize => 18.0;
  EdgeInsets get _marginContainerAll =>
      EdgeInsets.all(ApplicationConstants.instance.kChildContainerMargin);
  //EdgeInsets get _cardPaddingAll => const EdgeInsets.all(10.0);
}
