import 'package:flutter/material.dart';
import 'package:flutter_bloc/flutter_bloc.dart';
import 'package:intl/intl.dart';
import 'package:hamlemuhcrm/features/calendar/cubit/calendar_cubit.dart';
import 'package:hamlemuhcrm/features/calendar/cubit/calendar_state.dart';
import '../../../core/extensions/responsive_extension.dart';
import '../../../core/constants/color_constants.dart';
import '../../auth/login/cubit/login_cubit.dart';

final class GestureCalendarWidget extends StatefulWidget {
  const GestureCalendarWidget({super.key});

  @override
  // ignore: library_private_types_in_public_api
  _GestureCalendarWidgetState createState() => _GestureCalendarWidgetState();
}

class _GestureCalendarWidgetState extends State<GestureCalendarWidget> {
  final DateFormat _dateFormat = DateFormat.yMd('tr');
  String selectedDate = '';

  List<DateTime> _getWeeklyDates(DateTime currentDate) {
    final List<DateTime> dates = [];

    // 3 gün öncesi
    for (int i = -3; i < 0; i++) {
      dates.add(currentDate.add(Duration(days: i)));
    }

    // Bugün ve sonraki 3 gün
    for (int i = 0; i < 4; i++) {
      dates.add(currentDate.add(Duration(days: i)));
    }

    return dates;
  }

  List<String> _getWeekDayNames(DateTime currentDate) {
    final List<String> weekDayNames = [];
    final DateFormat dateFormat = DateFormat.E('tr');

    // 3 gün öncesi
    for (int i = -3; i < 0; i++) {
      final DateTime date = currentDate.add(Duration(days: i));
      weekDayNames.add(dateFormat.format(date));
    }

    // Bugün
    weekDayNames.add('Bugün');

    // Sonraki 3 gün
    for (int i = 1; i < 4; i++) {
      final DateTime date = currentDate.add(Duration(days: i));
      weekDayNames.add(dateFormat.format(date));
    }

    return weekDayNames;
  }

  Widget _buildDayWidget(DateTime date, bool isToday, String weekDayName) {
    final TextStyle textStyle =
        isToday ? const TextStyle(color: Colors.white) : const TextStyle();

    return Expanded(
      child: InkWell(
        onTap: () {
          // Güne tıklandığında yapılacak işlemler
          // Tıklanan günün DateTime değerini kullanabilirsiniz
          selectedDate = date.toString();
          //selectedDate.substring(0, 10) => gönderilecek gün
          debugPrint(selectedDate.substring(0, 10));
          context.read<CalendarCubit>().filterMeetings(
              context.read<LoginCubit>().token,
              context.read<LoginCubit>().id,
              selectedDate);
          //bu substring sayesinde gelen listemize filtreleme atarak o günkü toplantıları getirebiliriz
        },
        child: Container(
          padding: EdgeInsets.all(context.dynamicWidth(0.01)),
          decoration: BoxDecoration(
              color: isToday
                  ? ColorsConstants.instance.baseTheme
                  : Colors.transparent,
              borderRadius: BorderRadius.circular(10)),
          child: Column(
            children: [
              Text(
                weekDayName,
                style: textStyle,
              ),
              const SizedBox(height: 13),
              Text(
                _dateFormat.format(date).substring(0, 2),
                style: textStyle,
              ),
            ],
          ),
        ),
      ),
    );
  }

  @override
  Widget build(BuildContext context) {
    final DateTime now = DateTime.now();
    final List<DateTime> weeklyDates = _getWeeklyDates(now);
    final List<String> weekDayNames = _getWeekDayNames(now);

    return Container(
      height: context.dynamicHeight(0.2),
      decoration: const BoxDecoration(
        //dış çerçeve
        color: Colors.white,
        borderRadius: BorderRadius.all(Radius.circular(5)),
      ),
      alignment: Alignment.bottomCenter,
      child: Column(
        children: [
          Padding(
            padding: const EdgeInsets.all(15.0),
            child: Row(
              mainAxisAlignment: MainAxisAlignment.spaceBetween,
              children: [
                Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    BlocBuilder<CalendarCubit, CalendarState>(
                      builder: (context, state) {
                        return Text(
                          state.secilenGun,
                        );
                      },
                    ),
                    const Text('0 Toplantı'),
                  ],
                ),
                const Icon(
                  Icons.calendar_today_rounded,
                  size: 20,
                  color: Colors.blue,
                )
              ],
            ),
          ),
          const Spacer(),
          Container(
            padding: EdgeInsets.only(bottom: context.dynamicWidth(0.03)),
            decoration: const BoxDecoration(
              //dış çerçeve
              borderRadius: BorderRadius.all(Radius.circular(5)),
            ),
            child: Column(
              mainAxisAlignment: MainAxisAlignment.end,
              children: [
                Row(
                  children: weeklyDates.asMap().entries.map((entry) {
                    final int index = entry.key;
                    final DateTime date = entry.value;
                    final bool isToday = date.day == now.day &&
                        date.month == now.month &&
                        date.year == now.year;
                    final String weekDayName = weekDayNames[index];
                    return _buildDayWidget(date, isToday, weekDayName);
                  }).toList(),
                ),
              ],
            ),
          ),
        ],
      ),
    );
  }
}
