import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import 'package:flutter_bloc/flutter_bloc.dart';
import 'package:hamlemuhcrm/core/constants/font_styles.dart';
import 'package:hamlemuhcrm/core/extensions/responsive_extension.dart';
import 'package:hamlemuhcrm/features/auth/login/constants/login_constants.dart';
import 'package:hamlemuhcrm/features/auth/splash/constants/splash_constants.dart';
import 'package:hamlemuhcrm/features/auth/splash/cubit/splash_cubit.dart';
import 'package:hamlemuhcrm/features/auth/splash/animation/splash_lottie.dart';
import 'package:hamlemuhcrm/features/home/cubit/home_custom_cubit.dart';
import 'package:hamlemuhcrm/features/home/view/home_custom.dart';
import 'package:hamlemuhcrm/features/auth/login/cubit/login_cubit.dart';
import 'package:hamlemuhcrm/features/auth/login/view/login_view.dart';

import '../../../../core/constants/color_constants.dart';

class SplashView extends StatelessWidget {
  const SplashView({Key? key}) : super(key: key);

  @override
  Widget build(BuildContext context) {
    // Cubit'i sağlayalım
    return BlocProvider(
      create: (context) => SplashCubit(
          context.read<LoginCubit>(), context.read<HomeCustomCubit>()),
      child: BlocBuilder<SplashCubit, SplashState>(
        builder: (context, state) {
          // Cubit'in durumuna göre farklı widget'lar gösterelim
          switch (state) {
            case SplashState.initial:
              // Başlangıç durumunda, cubit'e istek atmasını söyleyelim
              context.read<SplashCubit>().getUserInfo();
              // Ve bir splash gösterelim
              return _columnScreen(Column(
                mainAxisAlignment: MainAxisAlignment.center,
                children: [
                  _splashLogo(context),
                  const Spacer(
                    flex: 2,
                  ),
                  _splashText,
                  const Spacer(
                    flex: 1,
                  ),
                  _loadingProgress,
                  const Spacer(
                    flex: 1,
                  ),
                  _versionFooter(context),
                ],
              ));
            case SplashState.loading:
              // Yükleme durumunda da, splash gösterelim
              return _columnScreen(Column(
                mainAxisAlignment: MainAxisAlignment.center,
                children: [
                  _splashLogo(context),
                  _emptyMiddleHeightSpace,
                  _splashText,
                  _emptyMiddleHeightSpace,
                  _loadingProgress,
                ],
              ));
            case SplashState.success:
              // Başarı durumunda, home custom sayfasına yönlendirelim
              return HomeCustom();
            case SplashState.failure:
              // Cubit sınıfında ki metotlara göre başarısızlık durumunda, loginscreen sayfasına yönlendirelim
              return const LoginView();
            case SplashState.timeout:
              // Cubit sınıfında ki metotlara göre başarısızlık durumunda, loginscreen sayfasına yönlendirelim
              return const LoginView();
            default:
              // Diğer durumlarda, bir hata mesajı gösterelim
              return _scaffoldMessage;
          }
        },
      ),
    );
  }

  Widget get _scaffoldMessage => const Scaffold(
        body: Center(
          child: Text('Teknik bir hata meydana geldi !'),
        ),
      );

  Widget get _emptyMiddleHeightSpace => const SizedBox(height: 25);

  Widget _columnScreen(Widget column) {
    return Scaffold(
      backgroundColor: ColorsConstants.instance.backgroundColor,
      appBar: AppBar(
        systemOverlayStyle: SystemUiOverlayStyle.dark,
        elevation: 0,
        backgroundColor: Colors.transparent,
      ),
      body: Center(child: column),
    );
  }

  Widget _splashLogo(BuildContext context) {
    return SizedBox(
        height: context.dynamicHeight(0.3), child: const SplashLottie());
  }

  Widget get _splashText =>
      Text(SplashConstants.splashText, style: FontStyles.instance.splashYazi);

  Widget get _loadingProgress => const CircularProgressIndicator(
        strokeWidth: 4.6,
      );

  Widget _versionFooter(BuildContext context) => Text(
        LoginConstants.verInfoText,
        style: FontStyles.instance.meetGeneralStyle,
      );
}
