import 'dart:async';

import 'package:connectivity_plus/connectivity_plus.dart';
import 'package:hamlemuhcrm/core/extensions/log_extension.dart';

typedef NetworkCallBack = void Function(NetworkResult result);

abstract class INetworkChangeManager {
  Future<NetworkResult> checkNetworkFirstTime();
  void handleNetworkChange(NetworkCallBack onChange);
  void dispose();
}

class NetworkChangeManager extends INetworkChangeManager {
  NetworkChangeManager() {
    _connectivity = Connectivity();
  }
  late final Connectivity _connectivity;
  StreamSubscription<ConnectivityResult>? _subscription;

  @override
  Future<NetworkResult> checkNetworkFirstTime() async {
    final connectivityResult = await _connectivity.checkConnectivity();
    return NetworkResult.checkConnectivityResult(connectivityResult);
  }

  @override
  void handleNetworkChange(NetworkCallBack onChange) {
    _subscription = _connectivity.onConnectivityChanged.listen((event) {
      'result : $event'.log();
      onChange.call(NetworkResult.checkConnectivityResult(event));
    });
  }

  @override
  void dispose() {
    _subscription?.cancel();
  }
}

enum NetworkResult {
  on,
  off;

  static NetworkResult checkConnectivityResult(ConnectivityResult result) {
    switch (result) {
      case ConnectivityResult.wifi:
      case ConnectivityResult.ethernet:
      case ConnectivityResult.mobile:
        return NetworkResult.on;
      case ConnectivityResult.bluetooth:
      case ConnectivityResult.none:
        return NetworkResult.off;
      case ConnectivityResult.vpn:
      case ConnectivityResult.other:
        return NetworkResult.off;
    }
  }
}
